<?php
/**
 * Modèle représentant un project dans le système de COMMUNIFY.
 * @author ARSENE TEKEU
 * @version 1.0
 * @date    Mai 2025
 * chaque project contient:
 * - un identifiant unique,
 * - un titre,
 * - une description,
 * - un objectif de financement,
 * - un montant actuel,
 * - une catégorie,
 * - une date de création,
 * - une date de fin,
 * - une image,
 * - un identifiant d'utilisateur,
 * - un statut (actif, financé, expiré)
 * @property int $id
 * @property string $titre
 * @property string $description
 * @property float $objectif
 * @property float $current_amount
 * @property string $categarie
 * @property DateTime $created_at
 * @property string $end_date
 * @property string $image
 * @property DateTime $id_users
 * @property int $statut
 * 
 */
require_once __DIR__ . '/../config/database.php';

class Projects
{
    /**
     * les attributs de la class Projets
     */
    private $id;
    private $titre;
    private $description;
    private $objectif;
    private $current_amount;
    private $categorie;
    private $created_at;
    private $end_date;
    private $image;
    private $id_users;
    private $statut;
    private $validate;

    public function getId()
    {
        return $this->id;
    }
    public function getCategorie()
    {
        return $this->categorie;
    }
    public function getTitre()
    {
        return $this->titre;
    }
    public function getDescription()
    {
        return $this->description;
    }
    public function getObjectif()
    {
        return $this->objectif;
    }
    public function getCurrentmontant()
    {
        return $this->current_amount;
    }
    public function getCreatedAt()
    {
        return $this->created_at;
    }
    public function getendDate()
    {
        return $this->end_date;
    }
    public function getImage()
    {
        return $this->image;
    }
    public function getIdUsers()
    {
        return $this->id_users;
    }
    //les setters
    /**
     * les setters
     * @param mixed $categorie
     * @return void
     */

    public function setCategorie($categorie)
    {
        $this->categorie = $categorie;
    }
    public function setTitre($titre)
    {
        $this->titre = $titre;
    }
    public function setDescription($description)
    {
        $this->description = $description;
    }
    public function setObjectif($objectif)
    {
        $this->objectif = $objectif;
    }
    public function setCurrentAmount($current_amount)
    {
        $this->current_amount = $current_amount;
    }

    /**
     * verifier le statut d'un projet
     * @return string
     */
    public function getStatut(): string
    {
        if ($this->current_amount >= $this->objectif)
            return 'financé';
        if ($this->isExpired())
            return 'expiré';
        return 'active';
    }
    // Autres méthodes utiles
    /**
     * methode qui verifie si le projet est expiré
     * @return bool
     */
    public function isExpired(): bool
    {
        if (!$this->end_date)
            return false;
        return (new DateTime()) > (new DateTime($this->end_date));
    }


    /**
     * methode de creation d'un projet
     * @param $titre
     * @param $description
     * @param $objectif
     * @param $categorie
     * @param $end_date
     * @param $image
     * @param $id_users

     */

    public function createProject($titre, $description, $objectif, $categorie, $end_date, $image, $id_users)
    {
        try {
            $conn = connectDB();
            $this->validateProjectData($titre, $description, $objectif, $categorie, $end_date, $image, $id_users);

            $stmt = $conn->prepare("INSERT INTO projects (titre, description,objectif,categorie,created_at,end_date, image,id_users) Values (:titre,:description,:objectif,:categorie,NOW(),:end_date,:image,:id_users)");

            // Nettoyage
            $params = [
                ':titre' => trim(strip_tags($titre)),
                ':description' => trim(strip_tags($description)),
                ':objectif' => (float) trim(strip_tags($objectif)),
                ':categorie' => trim(strip_tags($categorie)),
                ':end_date' => trim(strip_tags($end_date)),
                ':image' => trim(strip_tags($image)),
                ':id_users' => (int) trim(strip_tags($id_users))
            ];

            $stmt->execute($params);
            $this->id = $conn->lastInsertId();

            return true;

        } catch (PDOException $e) {
            throw new Exception("Project creation failed: " . $e->getMessage());
        }
    }

    /**
     * methode de modification d'un projet
     * @param $titre
     * @param $description
     * @param $objectif
     * @param $categorie
     * @param $end_date
     * @param $image
     * @param $id_users

     */
    public function updateProject(int $id, $titre, $description, $objectif, $categorie, $end_date, $image, $id_users): bool
    {
        try {
            $this->validateProjectData($titre, $description, $objectif, $categorie, $end_date, $image, $id_users);

            $query = "UPDATE projects SET 
                        titre = :titre, 
                        description = :description, 
                        objectif = :objectif, 
                        categorie = :categorie, 
                        end_date = :end_date, 
                        image = :image, 
                        id_users = :id_users 
                      WHERE id = :id";

            $params = [
                ':titre' => trim(strip_tags($titre)),
                ':description' => trim(strip_tags($description)),
                ':objectif' => trim(strip_tags($objectif)),
                ':categorie' => trim(strip_tags($categorie)),
                ':end_date' => trim(strip_tags($end_date)),
                ':image' => trim(strip_tags($image)),
                ':id_users' => trim(strip_tags($id_users)),
                ':id' => $id
            ];

            $conn = connectDB();
            $stmt = $conn->prepare($query);
            return $stmt->execute($params);

        } catch (PDOException $e) {
            throw new Exception("Échec de la mise à jour du projet.");
        }
    }
    /**
     * methode de suppression d'un projet
     * @param $id
     * @return true ou exception
     */

    public function deletes(int $id): bool
    {
        try {
            $conn = connectDB();
            $conn->beginTransaction();

            // Supprimer les contributions associées
            $stmt = $conn->prepare("DELETE FROM contributions WHERE id_projects = :id");
            $stmt->execute([':id' => $id]);

            // Supprimer le projet
            $stmt = $conn->prepare("DELETE FROM projects WHERE id = :id");
            $stmt->execute([':id' => $id]);

            $conn->commit();
            return true;
        } catch (PDOException $e) {
            if ($conn->inTransaction()) {
                $conn->rollBack();
            }
            throw new Exception("Échec de la suppression du projet.");
        }
    }


    // Méthodes métier


    /**
     * recuperer tous les projets et/ou en fonction de leur categorie,
     * du nombre de projet
     * @param mixed $category
     * @param mixed $limit
     * @param mixed $offset
     * @throws \Exception
     * @return array
     */
    public static function getAll($category = null) {
        $conn = connectDB();
        try {
            $query = "SELECT p.*, u.nom as porteur_project,
                     COUNT(DISTINCT c.id_users) as nombre_donateurs,
                     SUM(c.montant) as total_contribuer
                     FROM projects p 
                     JOIN users u ON p.id_users = u.id
                     LEFT JOIN contributions c ON p.id = c.id_projects
                     WHERE p.statut != 'supprimé'";
            
            $params = [];
            
            if ($category) {
                $query .= " AND p.categorie = :category";
                $params[':category'] = $category;
            }
            
            $query .= " GROUP BY p.id, p.titre, p.description, p.objectif, p.categorie, 
                       p.created_at, p.end_date, p.image, p.id_users, u.nom
                       ORDER BY p.created_at DESC";
            
            $stmt = $conn->prepare($query);
            
            if (!empty($params)) {
                foreach ($params as $key => $value) {
                    $stmt->bindValue($key, $value);
                }
            }
            
            $stmt->execute();
            $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if ($projects === false) {
                throw new Exception("Erreurs lors de la recuperations des projects");
            }
            
            return $projects;
        } catch (PDOException $e) {
            throw new Exception("Erreurs lors de la recuperations des projects");
        }
    }



    /**
     * recuperer une project en fonction de son identifiant
     * @param $id
     * @return $project
     */
    public function getProjectById(int $id): ?array
    {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("SELECT * FROM projects Where id=:id and statut != 'supprimé'");
            $stmt->execute([':id' => $id]);
            $project = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($project) {
                return $project;
            }
            return null;
        } catch (PDOException $e) {
            throw new Exception("project non existant !");
        }
    }

    /**
     * valider ou verifier l'existence des données avant insertion
     * @param mixed $titre
     * @param mixed $description
     * @param mixed $objectif
     * @param mixed $categorie
     * @param mixed $end_date
     * @param mixed $image
     * @param mixed $id_users
     * @throws \InvalidArgumentException
     * @return void
     */
    private function validateProjectData($titre, $description, $objectif, $categorie, $end_date, $image, $id_users): void
    {
        $data = [
            'titre' => $titre,
            'description' => $description,
            'objectif' => $objectif,
            'categorie' => $categorie,
            'end_date' => $end_date,
            'image' => $image,
            'id_users' => $id_users
        ];
        foreach ($data as $field => $value) {
            if (empty($value)) {
                throw new InvalidArgumentException("$field est requis");
            }
        }

        // Validation supplémentaire
        if (!is_numeric($objectif) || $objectif <= 0) {
            throw new InvalidArgumentException("le montant $objectif doit etre un nombre positif");
        }
    }

    /**
     * methode qui recuperer les projet en vedette
     */

    public static function getFeaturedTop10(): array
    {
        $db = connectDB();
        try {
            // Étape 1 : Mettre tous les projets en vedette à FALSE
            $db->exec("UPDATE projects SET vedette = FALSE");

            // Étape 2 : Trouver les 10 projets avec le plus grand nombre de contributeurs
            $stmtTop = $db->prepare("
            SELECT p.id
            FROM projects p
            LEFT JOIN contributions c ON p.id = c.id_projects
            GROUP BY p.id
            ORDER BY COUNT(DISTINCT c.id_users) DESC
            LIMIT 3
        ");
            $stmtTop->execute();
            $topProjects = $stmtTop->fetchAll(PDO::FETCH_COLUMN);

            // Étape 3 : Mettre ces projets en vedette
            if (!empty($topProjects)) {
                $placeholders = implode(',', array_fill(0, count($topProjects), '?'));
                $stmtUpdate = $db->prepare("UPDATE projects SET vedette = TRUE WHERE id IN ($placeholders)");
                $stmtUpdate->execute($topProjects);
            }

            // Étape 4 : Récupérer les projets mis en vedette avec les infos du créateur
            $stmt = $db->prepare("
            SELECT p.*, u.nom as porteur_project,
                   COUNT(DISTINCT c.id_users) as contributors_count,
                   SUM(c.montant) as total_contribuer
            FROM projects p
            JOIN users u ON p.id_users = u.id
            LEFT JOIN contributions c ON p.id = c.id_projects
            WHERE p.vedette = TRUE
            GROUP BY p.id, u.nom
            ORDER BY contributors_count DESC
        ");
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }


    /**
     * top 10 des meilleurs projets
     * @return array
     */
    public static function getFeaturedTop3()
    {
        $db = connectDB();
        try {
            // Étape 1 : Mettre tous les projets en vedette à FALSE
            $db->exec("UPDATE projects SET vedette = FALSE");

            // Étape 2 : Trouver les 10 projets les mieux financés
            $stmtTop = $db->prepare("
                SELECT p.id
                FROM projects p
                LEFT JOIN contributions c ON p.id = c.id_projects
                GROUP BY p.id
                ORDER BY SUM(c.montant) DESC
                LIMIT 3
            ");
            $stmtTop->execute();
            $topProjects = $stmtTop->fetchAll(PDO::FETCH_COLUMN);

            // Étape 3 : Mettre ces 10 projets en vedette
            if (!empty($topProjects)) {
                $placeholders = implode(',', array_fill(0, count($topProjects), '?'));
                $stmtUpdate = $db->prepare("UPDATE projects SET vedette = TRUE WHERE id IN ($placeholders)");
                $stmtUpdate->execute($topProjects);
            }

            // Étape 4 : Récupérer les projets mis en vedette avec les infos du créateur
            $stmt = $db->prepare("        
                SELECT p.*, u.nom as porteur_project,
                       COUNT(c.id) as contributors_count,
                       SUM(c.montant) as total_contribuer
                FROM projects p
                JOIN users u ON p.id_users = u.id
                LEFT JOIN contributions c ON p.id = c.id_projects
                WHERE p.vedette = TRUE
                GROUP BY p.id, u.nom
                ORDER BY total_contribuer DESC
            ");
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
    /**
     * la methode getDaysRemaining() retourne le nombre de jours restants avant la date de fin d'un projet.
     * @return $nombreDay
     */

    public function getDaysRemaining(): int
    {
        if (!$this->end_date)
            return 0;

        $now = new DateTime();
        $end_date = new DateTime($this->end_date);

        return $now < $end_date ? $end_date->diff($now)->days : 0;
    }

    public function getDaysRemainingText(): string
    {
        if (!$this->end_date)
            return "Date de fin non définie";

        $now = new DateTime();
        $end_date = new DateTime($this->end_date);

        if ($now < $end_date) {
            $diff = $end_date->diff($now)->days;
            return $diff === 1 ? "1 jour restant" : "$diff jours restants";
        }

        return "Projet terminé";
    }

    /**
     * ajouter une contributions a un projet
     * 
     * @param float $montant
     * @param int $id_users
     * @param mixed $message
     * @throws \Exception
     * @return bool
     */
    public function addContribution(float $montant, int $id_users, $message = ''): bool
    {
        try {
            $conn = connectDB();
            $conn->beginTransaction();

            // Vérifier si le projet est encore actif
            if ($this->getStatut() !== 'active') {
                throw new Exception("Le projet n'accepte plus de contributions.");
            }

            // Ajouter la contribution
            $stmt = $conn->prepare("INSERT INTO contributions 
                ( montant,message, created_at, id_users, id_projects) 
                VALUES (:montant,:message, NOW(), :id_users, :id_projects)");
            $stmt->execute([
                ':montant' => $montant,
                ':message' => $message,
                ':id_users' => $id_users,
                ':id_projects' => $this->id,

            ]);

            // Mettre à jour le montant actuel
            $stmt = $conn->prepare("UPDATE projects 
                SET current_amount = current_amount + :montant 
                WHERE id = :id");
            $stmt->execute([
                ':montant' => $montant,
                ':id' => $this->id
            ]);

            // Vérifier si le projet est financé
            if (($this->current_amount + $montant) >= $this->objectif) {
                $stmt = $conn->prepare("UPDATE projects SET statut = 'financé' WHERE id = :id");
                $stmt->execute([':id' => $this->id]);
            }

            $conn->commit();
            return true;
        } catch (Exception $e) {
            if ($conn->inTransaction()) {
                $conn->rollBack();
            }
            throw $e;
        }
    }






    public function deleteProject($id){
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("Update projects set statut='supprimé' where id = :id");
            $stmt->execute([':id' => $id]);
            return true;
        } catch (PDOException $e) {
            throw new Exception("Échec de la suppression du projet.");
        }
    }


}

?>