<?php
/**
 * Class ContributionController
 * 
 * @author  ARSENE TEKEU
 * @version 1.0
 * @date   Mai 2025
 */
// controllers/ContributionController.php

require_once __DIR__ . '/../models/Contributions.php';
require_once __DIR__ . '/../models/Projects.php';

class ContributionController {
    
    /**
     * Afficher le formulaire de contribution
     * @param mixed $id_projects
     * @return void
     */
    public function showContributionForm($id_projects) {
        // Vérifier si l'utilisateur est connecté
        if (!isset($_SESSION['user_id'])) {
            $_SESSION['error_message'] = "Vous devez être connecté pour contribuer à un projet";
            header('Location: index.php?action=login');
            exit;
        }
        
        $project = new Projects();
        $projectDetails = $project->getProjectById($id_projects);

        
        if (!$projectDetails) {
            $_SESSION['errors'] = "Ce projet n'existe pas";
            header('Location: index.php');
            exit;
        }
        
        // Vérifier si le projet est toujours en cours
        if ($project->isExpired() ) {
            $_SESSION['errors'] = "Ce projet est terminé, vous ne pouvez plus y contribuer";
            header('Location: index.php?action=project&id=' . $id_projects);
            exit;
        }
        
        include __DIR__ . '/../views/contributionView.php';
    }
    
    /**
     * Traiter la contribution
     * @param mixed $id_projects
     * @return never
     */
    public function processContribution($id_projects){
        // Vérification de la session
        if (!isset($_SESSION['user_id'])) {
            $_SESSION['redirect_url'] = "index.php?action=project&id=" . $id_projects;
            $_SESSION['error_message'] = "Connectez-vous pour contribuer";
            header('Location: index.php?action=login');
            exit;
        }
        
        $project = new Projects();
        $projectDetail = $project->getProjectById($id_projects);
        // Vérification du projet
        if (!$projectDetail || $projectDetail['statut'] !== 'active') {
            $_SESSION['errors'] = "Projet ".$project->getStatut();
            header('Location: index.php');
            exit;
        }
        // Vérification de la méthode
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $_SESSION['errors'] = "Méthode non autorisée";
            header('Location: index.php?action=project&id=' . $id_projects);
            exit;
        }
        // Validation des données
        $montant = filter_input(INPUT_POST, 'montant', FILTER_VALIDATE_FLOAT);
        $message = isset($_POST['message']) ? trim($_POST['message']) : '';
        $mode_payement = isset($_POST['mode_payement']) ? trim($_POST['mode_payement']) : null;
        $telephone = isset($_POST['telephone']) ? trim($_POST['telephone']) : null;
        $id_users= $_SESSION['user_id'];
        $errors = [];
    
        if ($montant === false || $montant <= 0) {
            $errors[] = "Montant invalide";
        }
        if (empty($mode_payement)) {
            $errors[] = "mode de paiement requis";
        }
        if (empty($telephone)) {
            $errors[] = "numéro de télephone requis";
        }
           
        if (empty($errors)) {
            $contribution = new Contributions(); 
            if ($contribution->create($montant, $mode_payement,$telephone, $id_users, $id_projects, $message)) {
                $_SESSION['success'] = "Merci pour votre contribution de " . number_format($montant, 2) . " FCFA !";
                header('Location: index.php?action=project&id=' . $id_projects);
                exit;
            } else {
                $errors[] = "Erreur lors de la contribution";
            }
        }
    
        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old_input'] = [
                'montant' => $_POST['montant'],
                'message' => $_POST['message'] ?? ''
            ];
             header('Location: index.php?action=contribute&id=' . $id_projects);
        exit;
        }
    
        
    }
    
    /**
     * Afficher toutes les contributions d'un utilisateur showUserContributions
     * @return void
     */
    public function showUserContributions() {
        // Vérifier si l'utilisateur est connecté
        if (!isset($_SESSION['user_id'])) {
            $_SESSION['errors_message'] = "Vous devez être connecté pour voir vos contributions";
            header('Location: index.php?action=login');
            exit;
        }
        
        $id_users = $_SESSION['user_id'];
        $contribution = new Contributions();
        $contributions = $contribution->getByUser($id_users);
        
        include __DIR__ . '/../views/templates/header.php';
        include __DIR__ . '/../views/my-contributions.php';
        include __DIR__ . '/../views/templates/footer.php';
    }
    
    /**
     * Afficher toutes les contributions pour un projet (Admin) showProjectContributions
     * @param mixed $id_projects
     * @return void
     */
    public function showProjectContributions($id_projects) {
        // Vérifier si l'utilisateur est connecté et est administrateur
        if (!isset($_SESSION['user_id']) || !$_SESSION['is_admin']) {
            $_SESSION['error_message'] = "Accès non autorisé";
            header('Location: index.php');
            exit;
        }
        
        $project = new Projects();
        $projectDetails = $project->getProjectById($id_projects);
        
        if (!$projectDetails) {
            $_SESSION['error_message'] = "Ce projet n'existe pas";
            header('Location: index.php');
            exit;
        }
        
        $contributions = Contributions::getContributionByProject($id_projects);
        
        include __DIR__ . '/../views/templates/header.php';
        include __DIR__ . '/../views/project-details.php';
        include __DIR__ . '/../views/templates/footer.php';
    }
    
}
?>