<?php
/**
 * Modèle représentant une contribution dans le système de COMMUNIFY.
 * @author ARSENE TEKEU 
 * @version 1.0.0
 * @date    Mai 2025
 * Chaque contribution contient :
 * - un identifiant unique,
 * - un montant,
 * - un message facultatif,
 * - une date de création,
 * - une référence à l’utilisateur et au projet concernés,
 * - les informations de paiement.
 *@property int $id
 * @property float $montant
 * @property string|null $message
 * @property datetime $created_at
 * @property int $id_users
 * @property int $id_projects
 * @property string $mode_payement
 * @property string $telephone
 */
require_once __DIR__ . '/../config/database.php';
require_once __DIR__."/../models/Projects.php";

class Contributions{
    private $id;
    private $montant;
    private $message;
    private $created_at;
    private $id_users;
    private $id_projects;
    private $mode_payement;
    private $telephone;

    /**
     * Créer une nouvelle contribution
     * @param mixed $montant
     * @param mixed $mode_payement
     * @param mixed $telephone
     * @param mixed $id_users
     * @param mixed $id_projects
     * @param mixed $message
     * @throws \Exception
     * @return bool
     */
    public function create($montant, $mode_payement,$telephone, $id_users, $id_projects, $message = '') {
        $conn = connectDB();
        
        try {
            // Validation supplémentaire du montant
            if ($montant <= 0) {
                throw new Exception("Montant invalide");
            }
            
            // Commence la transaction
            $conn->beginTransaction();
            
            // 1. Insérer la contribution
            $stmt = $conn->prepare("INSERT INTO contributions (montant, message,mode_payement,telephone ,created_at, id_users, id_projects) 
                                   VALUES (:montant, :message,:mode_payement,:telephone, NOW(), :id_users, :id_projects)");
            $stmt->bindParam(':montant', $montant, PDO::PARAM_INT);
            $stmt->bindParam(':message', $message, PDO::PARAM_STR);
            $stmt->bindParam(':mode_payement', $mode_payement, PDO::PARAM_STR);
            $stmt->bindParam(':telephone', $telephone, PDO::PARAM_STR);
            $stmt->bindParam(':id_users', $id_users, PDO::PARAM_INT);
            $stmt->bindParam(':id_projects', $id_projects, PDO::PARAM_INT);
            $stmt->execute();
            
            
            // 2. Mettre à jour le projet
            $update = $conn->prepare("UPDATE projects 
                                   SET current_amount = current_amount + :montant,
                                       created_at = NOW()
                                   WHERE id = :id_projects");
            $update->bindParam(':montant', $montant, PDO::PARAM_INT);
            $update->bindParam(':id_projects', $id_projects, PDO::PARAM_INT);
            $update->execute();
            
            // 3. Vérifier si l'objectif est atteint
            $select = $conn->prepare("SELECT objectif, current_amount FROM projects WHERE id = :id_projects");
            $select->bindParam(':id_projects', $id_projects, PDO::PARAM_INT);
            $select->execute();
            $project = $select->fetch(PDO::FETCH_ASSOC);
            
            // Si l'objectif est atteint, marquer le projet comme financé
            
            if ($project['current_amount'] >= $project['objectif']) {
                $stmt = $conn->prepare("UPDATE projects SET statut = 'financé' WHERE id = :id_projects");
                $stmt->bindParam(':id_projects', $id_projects, PDO::PARAM_INT);
                $stmt->execute();
            }
            // Commit de la transaction si tout a réussi
            $conn->commit();
            return true;
    
        } catch (Exception $e) {
            
            // Annulation de la transaction en cas d'erreur
            if($conn->inTransaction()){
                $conn->rollBack();
            }
            error_log("Erreur de contribution: " . $e->getMessage());
            return false;
        }
    }
    
    // Récupérer toutes les contributions pour un projet
    /**
     * Récupérer toutes les contributions pour un projet getContributionByProject
     * @param mixed $id_projects
     * @return array
     */
    public static function getContributionByProject($id_projects) {
    $conn = connectDB();

    try {
        // 1. Récupérer les contributions
        $stmt = $conn->prepare("SELECT c.*, u.nom,u.created_at as dates
                               FROM contributions c 
                               JOIN users u ON c.id_users = u.id 
                               WHERE c.id_projects = :id_projects 
                               ORDER BY c.created_at DESC");
        $stmt->bindParam(':id_projects', $id_projects);
        $stmt->execute();
        $contributions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // 2. Compter les donateurs uniques
        $stmtCount = $conn->prepare("SELECT COUNT(DISTINCT id_users) AS total_donateurs 
                                     FROM contributions 
                                     WHERE id_projects = :id_projects");
        $stmtCount->bindParam(':id_projects', $id_projects);
        $stmtCount->execute();
        $donorCount = $stmtCount->fetch(PDO::FETCH_ASSOC)['total_donateurs'];

        // 3. Retourner les données
        return [
            'contributions' => $contributions,
            'donateurs_count' => $donorCount
        ];
    } catch (PDOException $e) {
        return [
            'contributions' => [],
            'donateurs_count' => 0
        ];
    }
}


    /**
     * recuperer les contributions en fonctions de l'identifiants des contributeurs
     * Summary of getByUser
     * @param mixed $id_users
     * @return array
     */
    public static function getByUser($id_users) {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare('
                SELECT c.*, p.titre
                FROM contributions c
                JOIN projects p ON c.id_projects = p.id
                WHERE c.id_users = :id_users
                ORDER BY c.created_at DESC');
            $stmt->bindParam(':id_users', $id_users, PDO::PARAM_INT);
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
 
    /**
     * Récupérer le nombre total de contributions pour un projet
     * Summary of getCountByProject
     * @param mixed $id_projects
     */
    public static function getCountByProject($id_projects) {
        $conn = connectDB();
        
        try {
            $stmt = $conn->prepare("SELECT COUNT(*) FROM contributions WHERE id_projects = :id_projects");
            $stmt->bindParam(':id_projects', $id_projects);
            $stmt->execute();
            
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }
    
    // Récupérer la contribution moyenne pour un projet
    /**
     * Récupérer la contribution moyenne pour un projet
     * Summary of getAverageByProject
     * @param mixed $id_projects
     */
    public static function getAverageByProject($id_projects) {
        $conn = connectDB();
        
        try {
            $stmt = $conn->prepare("SELECT AVG(montant) FROM contributions WHERE id_projects = :id_projects");
            $stmt->bindParam(':id_projects', $id_projects);
            $stmt->execute();
            
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }
    /**
     * Summary of getSumByProject
     * @param mixed $id_projects
     */
    public static function getSumByProject($id_projects) {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("SELECT SUM(montant) FROM contributions WHERE id_projects = :id_projects");
            $stmt->bindParam(':id_projects', $id_projects);
            $stmt->execute();
            
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            return 0;
        }
    }     

    public function deleteByProject($projectId) {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("DELETE FROM contributions WHERE id_projects = :project_id");
            $stmt->bindParam(':project_id', $projectId);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            return false;
        }
    }














}
// Fin de la classe Contributions
?>